﻿using System;
using System.Web.Mvc;
using System.Linq;
using MVC_App.Models;
using MVC_App.Validation;

namespace MVC_App.Controllers
{
	public class HomeController : Controller
	{
		[HttpGet]
		public ActionResult EditOrder()
		{
			return View(BuildDummyBestellung());
		}

		static BestellungForm BuildDummyBestellung()
		{
			return new BestellungForm
					{
						Lieferadresse = new LieferadresseForm(),
						Kreditkarte = new KreditkarteForm(),
						Pizza = new PizzaForm
											{
												Größe = PizzaGrößen.Mittel,
											}
					};
		}

		[HttpPost]
		public ActionResult SubmitOrder(BestellungForm bestellung)
		{
			var lieferzeitpunkt = ValidateLieferzeitpunkt(bestellung);

			if (ModelState.IsValid && lieferzeitpunkt != null)
			{
				// TODO: Ergebnisse verarbeiten
				return View("Success");
			}

			return View("EditOrder", bestellung);
		}

		private DateTime? ValidateLieferzeitpunkt(BestellungForm bestellung)
		{
			SimpleTime time;
			switch (bestellung.LieferzeitpunktArt)
			{
				case Lieferzeitpunkte.Sofort:
					return DateTime.Now;

				case Lieferzeitpunkte.Heute:
					time = ValidateUhrzeit(bestellung);
					if (time == null)
						return null;

					var today = DateTime.Now;
					return new DateTime(today.Year, today.Month, today.Day, time.Hour, time.Minute, 0);

				case Lieferzeitpunkte.Am:
					time = ValidateUhrzeit(bestellung);
					var nDate = ValidateDatum(bestellung);

					if (time == null || nDate == null)
						return null;

					var date = nDate.Value;
					return new DateTime(date.Year, date.Month, date.Day, time.Hour, time.Minute, 0);

				default:
					throw new ArgumentOutOfRangeException();
			}
		}

		private SimpleTime ValidateUhrzeit(BestellungForm bestellung)
		{
			var zeit = bestellung.Lieferzeitpunkt.Uhrzeit;
			if (zeit != null)
			{
				var parts = zeit.Split(':');
				if (parts.Length == 2)
				{
					int hour;
					int minute;
					if (int.TryParse(parts[0], out hour)
						&& int.TryParse(parts[1], out minute))
					{
						if (minute >= 0 && minute <= 60
							&& hour >= 0 && hour <= 24)
						{
							return new SimpleTime(hour, minute);
						}
					}
				}
			}

			ModelState.AddModelError("Lieferzeitpunkt.Uhrzeit", "Bitte geben Sie eine gültige Uhrzeit für die Lieferung an.");
			return null;
		}

		private DateTime? ValidateDatum(BestellungForm bestellung)
		{
			var datumString = bestellung.Lieferzeitpunkt.Datum;
			if (datumString != null)
			{
				DateTime datum;
				if (DateTime.TryParse(datumString, out datum))
				{
					return datum;
				}
			}

			ModelState.AddModelError("Lieferzeitpunkt.Datum", "Bitte geben Sie ein gültiges Datum für die Lieferung an.");
			return null;
		}
	}

	public class SimpleTime
	{
		public SimpleTime(int hour, int minute)
		{
			Hour = hour;
			Minute = minute;
		}

		public int Hour { get; private set; }
		public int Minute { get; private set; }
	}
}
